import { supabase, TABLES } from '@/config/supabase';
import {
  ShoppingList,
  ShoppingListItem,
  SellerMatchResult,
  Order,
  OrderItem,
  OrderStatus
} from './types';

export const shoppingListService = {
  // Create shopping list and items
  async createShoppingList(vendor_id: string, name: string, notes: string, items: Array<{ item_name: string; quantity: number; unit: string; notes?: string }>) {
    // 1. Ensure all items exist in global catalog
    const itemIds: Record<string, string> = {};
    for (const item of items) {
      // Try to find by name or synonym
      let { data: foundItem } = await supabase.from('items').select('id').ilike('name', item.item_name).single();
      if (!foundItem) {
        // Create new canonical item
        const { data: newItem } = await supabase.from('items').insert([{ name: item.item_name }]).select().single();
        itemIds[item.item_name] = newItem.id;
      } else {
        itemIds[item.item_name] = foundItem.id;
      }
    }
    // 2. Create shopping list
    const { data: list, error: listError } = await supabase.from('shopping_lists').insert([{ vendor_id, name, notes }]).select().single();
    if (listError) return { error: listError };
    // 3. Create shopping list items
    const itemsToInsert = items.map(item => ({
      shopping_list_id: list.id,
      item_id: itemIds[item.item_name],
      item_name: item.item_name,
      quantity: item.quantity,
      unit: item.unit,
      notes: item.notes || null
    }));
    const { error: itemsError } = await supabase.from('shopping_list_items').insert(itemsToInsert);
    if (itemsError) return { error: itemsError };
    return { data: list };
  },

  // Seller matching
  async matchSellersForList(list_id: string) {
    // 1. Get all items in the list
    const { data: listItems, error: listItemsError } = await supabase.from('shopping_list_items').select('item_id,item_name,quantity').eq('shopping_list_id', list_id);
    if (listItemsError) return { error: listItemsError };
    // 2. For each item, find sellers with inventory
    const sellerMatches: Record<string, { seller: any; matched_items: any[] }> = {};
    for (const item of listItems) {
      // Find sellers with this item (by item_id)
      const { data: inventory } = await supabase.from('seller_inventory').select('seller_id,price,stock_level').eq('item_id', item.item_id);
      for (const inv of inventory || []) {
        // Get seller info
        if (!sellerMatches[inv.seller_id]) {
          const { data: seller } = await supabase.from('sellers').select('id,shop_name,delivery_options').eq('id', inv.seller_id).single();
          sellerMatches[inv.seller_id] = { seller, matched_items: [] };
        }
        sellerMatches[inv.seller_id].matched_items.push({ item_name: item.item_name, price: inv.price, stock_level: inv.stock_level });
      }
    }
    // 3. Build match results
    const results: SellerMatchResult[] = Object.values(sellerMatches).map(match => ({
      seller: match.seller,
      match_count: match.matched_items.length,
      matched_items: match.matched_items
    }));
    // Sort by match_count descending
    results.sort((a, b) => b.match_count - a.match_count);
    return { data: results };
  },

  // Create order from selected seller
  async createOrder(vendor_id: string, shopping_list_id: string, seller_id: string, items: Array<{ item_id: string; quantity: number }>) {
    // 1. Get unit price for each item from seller_inventory
    let total_price = 0;
    const orderItemsToInsert: any[] = [];
    for (const item of items) {
      const { data: inv } = await supabase.from('seller_inventory').select('price,unit').eq('seller_id', seller_id).eq('item_id', item.item_id).single();
      if (!inv) return { error: { message: `Seller does not have item ${item.item_id}` } };
      const item_total = inv.price * item.quantity;
      total_price += item_total;
      orderItemsToInsert.push({
        item_id: item.item_id,
        quantity: item.quantity,
        unit: inv.unit,
        unit_price: inv.price,
        total_price: item_total
      });
    }
    // 2. Create order
    const { data: order, error: orderError } = await supabase.from('orders').insert([{
      vendor_id,
      seller_id,
      shopping_list_id,
      total_price,
      status: 'pending'
    }]).select().single();
    if (orderError) return { error: orderError };
    // 3. Create order items
    const orderItems = orderItemsToInsert.map(item => ({
      ...item,
      order_id: order.id
    }));
    const { error: orderItemsError } = await supabase.from('order_items').insert(orderItems);
    if (orderItemsError) return { error: orderItemsError };
    return { data: order };
  },

  // Get orders for vendor
  async getOrders(vendor_id: string, status?: string) {
    let query = supabase.from('orders').select('*,seller:seller_id(id,shop_name,delivery_options),order_items:order_items(*)').eq('vendor_id', vendor_id);
    if (status) query = query.eq('status', status);
    const { data, error } = await query.order('created_at', { ascending: false });
    return { data, error };
  },

  // Update order status
  async updateOrderStatus(order_id: string, vendor_id: string, status: OrderStatus) {
    const { data, error } = await supabase.from('orders').update({ status }).eq('id', order_id).eq('vendor_id', vendor_id).select().single();
    return { data, error };
  }
}; 